/*
 * Copyright (C) 2010-2011 Freescale Semiconductor, Inc.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/io.h>
#include <asm/arch/mx53.h>
#include <asm/arch/mx53_pins.h>
#include <asm/arch/iomux.h>
#include <asm/errno.h>
#include <netdev.h>

#if CONFIG_I2C_MXC
#include <i2c.h>
#endif

#ifdef CONFIG_CMD_MMC
#include <mmc.h>
#include <fsl_esdhc.h>
#endif

#ifdef CONFIG_ARCH_MMU
#include <asm/mmu.h>
#include <asm/arch/mmu.h>
#endif

#ifdef CONFIG_GET_FEC_MAC_ADDR_FROM_IIM
#include <asm/imx_iim.h>
#endif

#ifdef CONFIG_CMD_CLOCK
#include <asm/clock.h>
#endif

#ifdef CONFIG_ANDROID_RECOVERY
#include "../common/recovery.h"
#include <part.h>
#include <ext2fs.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <ubi_uboot.h>
#include <jffs2/load_kernel.h>
#endif

DECLARE_GLOBAL_DATA_PTR;

#define RJK_DEBUG 1   // sets debug MAC address

//Crestron Product ID information
extern unsigned int gProdId;

static u32 system_rev;
static enum boot_device boot_dev;
const char *g_boot_reason = "unknown";

static int BatteryGood (int dwBootSOCThreshold, int dwBootVoltageThreshold);
static void OALDa9052ShutDown(void);

u32 get_board_rev_from_fuse(void)
{
	u32 board_rev = readl(IIM_BASE_ADDR + 0x878);

	return board_rev;
}

static inline void setup_boot_device(void)
{
	uint soc_sbmr = readl(SRC_BASE_ADDR + 0x4);
	uint bt_mem_ctl = (soc_sbmr & 0x000000FF) >> 4 ;
	uint bt_mem_type = (soc_sbmr & 0x00000008) >> 3;

	switch (bt_mem_ctl) {
	case 0x0:
		if (bt_mem_type)
			boot_dev = ONE_NAND_BOOT;
		else
			boot_dev = WEIM_NOR_BOOT;
		break;
	case 0x2:
		if (bt_mem_type)
			boot_dev = SATA_BOOT;
		else
			boot_dev = PATA_BOOT;
		break;
	case 0x3:
		if (bt_mem_type)
			boot_dev = SPI_NOR_BOOT;
		else
			boot_dev = I2C_BOOT;
		break;
	case 0x4:
	case 0x5:
		boot_dev = SD_BOOT;
		break;
	case 0x6:
	case 0x7:
		boot_dev = MMC_BOOT;
		break;
	case 0x8 ... 0xf:
		boot_dev = NAND_BOOT;
		break;
	default:
		boot_dev = UNKNOWN_BOOT;
		break;
	}
}

enum boot_device get_boot_device(void)
{
	return boot_dev;
}

u32 get_board_rev(void)
{
	return system_rev;
}

static inline void setup_soc_rev(void)
{
	int reg;

	/* Si rev is obtained from ROM */
	reg = __REG(ROM_SI_REV);

	switch (reg) {
	case 0x10:
		system_rev = 0x53000 | CHIP_REV_1_0;
		break;
	case 0x20:
		system_rev = 0x53000 | CHIP_REV_2_0;
		break;
	case 0x21:
		system_rev = 0x53000 | CHIP_REV_2_1;
		break;
	default:
		system_rev = 0x53000 | CHIP_REV_UNKNOWN;
	}
}

static inline void setup_board_rev(int rev)
{
	system_rev |= (rev & 0xF) << 8;
}

inline int is_soc_rev(int rev)
{
	return (system_rev & 0xFF) - rev;
}

#ifdef CONFIG_ARCH_MMU
void board_mmu_init(void)
{
	unsigned long ttb_base = PHYS_SDRAM_1 + 0x4000;
	unsigned long i;

	/*
	* Set the TTB register
	*/
	asm volatile ("mcr  p15,0,%0,c2,c0,0" : : "r"(ttb_base) /*:*/);

	/*
	* Set the Domain Access Control Register
	*/
	i = ARM_ACCESS_DACR_DEFAULT;
	asm volatile ("mcr  p15,0,%0,c3,c0,0" : : "r"(i) /*:*/);

	/*
	* First clear all TT entries - ie Set them to Faulting
	*/
	memset((void *)ttb_base, 0, ARM_FIRST_LEVEL_PAGE_TABLE_SIZE);
	/* Actual   Virtual  Size   Attributes          Function */
	/* Base     Base     MB     cached? buffered?  access permissions */
	/* xxx00000 xxx00000 */
	X_ARM_MMU_SECTION(0x000, 0x000, 0x010,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* ROM, 16M */
	X_ARM_MMU_SECTION(0x010, 0x010, 0x060,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* Reserved, 96M */
	X_ARM_MMU_SECTION(0x070, 0x070, 0x010,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* IRAM, 16M */
	X_ARM_MMU_SECTION(0x080, 0x080, 0x080,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* Reserved region + TZIC. 1M */
	X_ARM_MMU_SECTION(0x100, 0x100, 0x040,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* SATA */
	X_ARM_MMU_SECTION(0x140, 0x140, 0x040,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* Reserved, 64M */
	X_ARM_MMU_SECTION(0x180, 0x180, 0x080,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* IPUv3M */
	X_ARM_MMU_SECTION(0x200, 0x200, 0x200,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* GPU */
	X_ARM_MMU_SECTION(0x400, 0x400, 0x300,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* periperals */
	X_ARM_MMU_SECTION(0x700, 0x700, 0x200,
			ARM_CACHEABLE, ARM_BUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* CSD0 512M */
	X_ARM_MMU_SECTION(0x700, 0x900, 0x200,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* CSD0 512M */
	X_ARM_MMU_SECTION(0xB00, 0xB00, 0x200,
			ARM_CACHEABLE, ARM_BUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* CSD0 512M */
	X_ARM_MMU_SECTION(0xB00, 0xD00, 0x200,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* CSD0 512M */
	X_ARM_MMU_SECTION(0xF00, 0xF00, 0x07F,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* CS1 EIM control*/
	X_ARM_MMU_SECTION(0xF7F, 0xF7F, 0x001,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* NAND Flash buffer */
	X_ARM_MMU_SECTION(0xF80, 0xF80, 0x080,
			ARM_UNCACHEABLE, ARM_UNBUFFERABLE,
			ARM_ACCESS_PERM_RW_RW); /* iRam + GPU3D + Reserved */

	/* Workaround for arm errata #709718 */
	/* Setup PRRR so device is always mapped to non-shared */
	asm volatile ("mrc p15, 0, %0, c10, c2, 0" : "=r"(i) : /*:*/);
	i &= (~(3 << 0x10));
	asm volatile ("mcr p15, 0, %0, c10, c2, 0" : : "r"(i) /*:*/);

	/* Enable MMU */
	MMU_ON();
}
#endif

int dram_init(void)
{
	gd->bd->bi_dram[0].start = PHYS_SDRAM_1;
	gd->bd->bi_dram[0].size = PHYS_SDRAM_1_SIZE;
	gd->bd->bi_dram[1].start = PHYS_SDRAM_2;
	gd->bd->bi_dram[1].size = PHYS_SDRAM_2_SIZE;
	return 0;
}

/*
00 Selecting Pad: CSI0_DAT10 for Mode: ALT2.
01 Selecting Pad: CSI0_DAT11 for Mode: ALT2.
10 Selecting Pad: PATA_DIOW for Mode: ALT3.
11 Selecting Pad: PATA_DMACK for Mode: ALT3.
*/
static void setup_uart(void)
{
	switch (gProdId)
	{
		case TST900:
			/* UART1 RXD */
			mxc_request_iomux(MX53_PIN_ATA_DMACK, IOMUX_CONFIG_ALT3);
			mxc_iomux_set_pad(MX53_PIN_ATA_DMACK, 0x1E4);
			mxc_iomux_set_input(MUX_IN_UART1_IPP_UART_RXD_MUX_SELECT_INPUT, 0x3); // 11 Selecting Pad: PATA_DMACK for Mode: ALT3.

			/* UART1 TXD */
			mxc_request_iomux(MX53_PIN_ATA_DIOW, IOMUX_CONFIG_ALT3);
			mxc_iomux_set_pad(MX53_PIN_ATA_DIOW, 0x1E4);
			break;
		case TSW1050:
		case TSW750:
		case TSM750:
		case TSW550:
		case TST600:
		case TSR300:
		default:
		{
			/* UART1 RXD */
			mxc_request_iomux(MX53_PIN_CSI0_D11, IOMUX_CONFIG_ALT2);
			mxc_iomux_set_pad(MX53_PIN_CSI0_D11, 0x1E4);
			mxc_iomux_set_input(MUX_IN_UART1_IPP_UART_RXD_MUX_SELECT_INPUT, 0x1); // 01 Selecting Pad: CSI0_DAT11 for Mode: ALT2.

			/* UART1 TXD */
			mxc_request_iomux(MX53_PIN_CSI0_D10, IOMUX_CONFIG_ALT2);
			mxc_iomux_set_pad(MX53_PIN_CSI0_D10, 0x1E4);
			break;
		}
	}

}

#ifdef CONFIG_I2C_MXC
static void setup_i2c(unsigned int module_base)
{
	switch (module_base) {
	case I2C1_BASE_ADDR:
		/* i2c1 SDA */
		mxc_request_iomux(MX53_PIN_CSI0_D8,
				IOMUX_CONFIG_ALT5 | IOMUX_CONFIG_SION);
		mxc_iomux_set_input(MUX_IN_I2C1_IPP_SDA_IN_SELECT_INPUT,
				INPUT_CTL_PATH0);
		mxc_iomux_set_pad(MX53_PIN_CSI0_D8, PAD_CTL_SRE_FAST |
				PAD_CTL_ODE_OPENDRAIN_ENABLE |
				PAD_CTL_PKE_ENABLE | PAD_CTL_PUE_PULL |
				PAD_CTL_DRV_HIGH | PAD_CTL_100K_PU |
				PAD_CTL_HYS_ENABLE);
		/* i2c1 SCL */
		mxc_request_iomux(MX53_PIN_CSI0_D9,
				IOMUX_CONFIG_ALT5 | IOMUX_CONFIG_SION);
		mxc_iomux_set_input(MUX_IN_I2C1_IPP_SCL_IN_SELECT_INPUT,
				INPUT_CTL_PATH0);
		mxc_iomux_set_pad(MX53_PIN_CSI0_D9, PAD_CTL_SRE_FAST |
				PAD_CTL_ODE_OPENDRAIN_ENABLE |
				PAD_CTL_PKE_ENABLE | PAD_CTL_PUE_PULL |
				PAD_CTL_DRV_HIGH | PAD_CTL_100K_PU |
				PAD_CTL_HYS_ENABLE);
		break;
	case I2C2_BASE_ADDR:
		/* i2c2 SDA */
		mxc_request_iomux(MX53_PIN_KEY_ROW3,
				IOMUX_CONFIG_ALT4 | IOMUX_CONFIG_SION);
		mxc_iomux_set_input(MUX_IN_I2C2_IPP_SDA_IN_SELECT_INPUT,
				INPUT_CTL_PATH0);
		mxc_iomux_set_pad(MX53_PIN_KEY_ROW3,
				PAD_CTL_SRE_FAST |
				PAD_CTL_ODE_OPENDRAIN_ENABLE |
				PAD_CTL_DRV_HIGH | PAD_CTL_100K_PU |
				PAD_CTL_HYS_ENABLE);

		/* i2c2 SCL */
		mxc_request_iomux(MX53_PIN_KEY_COL3,
				IOMUX_CONFIG_ALT4 | IOMUX_CONFIG_SION);
		mxc_iomux_set_input(MUX_IN_I2C2_IPP_SCL_IN_SELECT_INPUT,
				INPUT_CTL_PATH0);
		mxc_iomux_set_pad(MX53_PIN_KEY_COL3,
				PAD_CTL_SRE_FAST |
				PAD_CTL_ODE_OPENDRAIN_ENABLE |
				PAD_CTL_DRV_HIGH | PAD_CTL_100K_PU |
				PAD_CTL_HYS_ENABLE);

		break;
	case I2C3_BASE_ADDR:
		break;
	default:
		printf("Invalid I2C base: 0x%x\n", module_base);
		break;
	}
}

void setup_pmic_voltages(void)
{
}

/* DA9053 I2C SDA stuck low issue: the I2C block in DA9053 may not correctly
 * receive a Power On Reset and device is in unknown state during start-up.
 * The only way to get the chip into known state before any communication
 * with the Chip via I2C is to dummy clock the I2C and bring it in a state
 * where I2C can communicate. Dialog suggested to provide 9 clock on SCL.
 * Dialog don't know the exact reason for the fault and assume it is because
 * some random noise or spurious behaviour.
 * This has to been done in host platform specific I2C driver during
 * start-up when the I2C is being configured at platform level to supply with
 * dummy 9 clock on SCL. Dialog I2C driver has no control to provide dummy 9
 * clock on SCL.
 */
#define I2C1_SDA_GPIO5_26_BIT_MASK  (1 << 26)
#define I2C1_SCL_GPIO5_27_BIT_MASK  (1 << 27)
void i2c_failed_handle(void)
{
	unsigned int reg, i, retry = 10;

	do {
		/* set I2C1_SDA as GPIO input */
		mxc_request_iomux(MX53_PIN_CSI0_D8, IOMUX_CONFIG_ALT1);
		reg = readl(GPIO5_BASE_ADDR + 0x4);
		reg &= ~I2C1_SDA_GPIO5_26_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + 0x4);

		/* set I2C1_SCL as GPIO output */
		mxc_request_iomux(MX53_PIN_CSI0_D9, IOMUX_CONFIG_ALT1);
		reg = readl(GPIO5_BASE_ADDR + 0x0);
		reg |= I2C1_SCL_GPIO5_27_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + 0x0);

		reg = readl(GPIO5_BASE_ADDR + 0x4);
		reg |= I2C1_SCL_GPIO5_27_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + 0x4);
		udelay(10000);

		for (i = 0; i < 10; i++) {
			reg = readl(GPIO5_BASE_ADDR + 0x0);
			reg |= I2C1_SCL_GPIO5_27_BIT_MASK;
			writel(reg, GPIO5_BASE_ADDR + 0x0);
			udelay(5000);

			reg = readl(GPIO5_BASE_ADDR + 0x0);
			reg &= ~I2C1_SCL_GPIO5_27_BIT_MASK;
			writel(reg, GPIO5_BASE_ADDR + 0x0);
			udelay(5000);
		}
		reg = readl(GPIO5_BASE_ADDR + 0x0);
		reg |= I2C1_SCL_GPIO5_27_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + 0x0);
		udelay(1000);

		reg = readl(GPIO5_BASE_ADDR + 0x8);
		if (reg & I2C1_SDA_GPIO5_26_BIT_MASK) {
			printf("***I2C1_SDA = hight***\n");
			return;
		} else {
			printf("***I2C1_SDA = low***\n");
		}
	} while (retry--);
}

int i2c_read_check(uchar chip, uint addr, int alen, uchar *buf, int len)
{
	int ret = 0;

	ret = i2c_read(chip, addr, alen, buf, len);
	if (ret == 0) {
		return 0;
	} else {
	i2c_failed_handle();
	setup_i2c(CONFIG_SYS_I2C_PORT);
	ret = i2c_read(chip, addr, alen, buf, len);
	if (ret != 0) {
		printf("[I2C-DA9053]read i2c fail\n");
		return -1;
	}
	return 0;
	}
}

int i2c_write_check(uchar chip, uint addr, int alen, uchar *buf, int len)
{
	int ret = 0;

	ret = i2c_write(chip, addr, alen, buf, len);
	if (ret == 0) {
		return 0;
	} else {
		i2c_failed_handle();
		setup_i2c(CONFIG_SYS_I2C_PORT);
		i2c_init(CONFIG_SYS_I2C_SPEED, CONFIG_SYS_I2C_SLAVE);
		ret = i2c_write(chip, addr, alen, buf, len);
		if (ret != 0) {
			printf("[I2C-DA9053]write i2c fail\n");
			return -1;
		}
		return 0;
	}
}
#endif

#if defined(CONFIG_DWC_AHSATA)
static void setup_sata_device(void)
{
	u32 *tmp_base =
		(u32 *)(IIM_BASE_ADDR + 0x180c);
	u32 reg;
	mxc_request_iomux(MX53_PIN_EIM_DA3, IOMUX_CONFIG_ALT1);

	/* GPIO_3_3 */
	reg = readl(GPIO3_BASE_ADDR);
	reg |= (0x1 << 3);
	writel(reg, GPIO3_BASE_ADDR);

	reg = readl(GPIO3_BASE_ADDR + 0x4);
	reg |= (0x1 << 3);
	writel(reg, GPIO3_BASE_ADDR + 0x4);

	udelay(1000);

	/* Set USB_PHY1 clk, fuse bank4 row3 bit2 */
	set_usb_phy1_clk();
	writel((readl(tmp_base) & (~0x7)) | 0x4, tmp_base);
}
#endif

#ifdef CONFIG_MXC_FEC
#ifdef CONFIG_GET_FEC_MAC_ADDR_FROM_IIM

#define RJK_DEBUG 1
int fec_get_mac_addr(unsigned char *mac)
{

#if RJK_DEBUG
	mac[0]=0x00;
	mac[1]=0xde;
	mac[2]=0xad;
	mac[3]=0xbe;
	mac[4]=0xef;
	mac[5]=0x00;

#else
	u32 *iim1_mac_base =
		(u32 *)(IIM_BASE_ADDR + IIM_BANK_AREA_1_OFFSET +
			CONFIG_IIM_MAC_ADDR_OFFSET);
	int i;

	for (i = 0; i < 6; ++i, ++iim1_mac_base)
		mac[i] = (u8)readl(iim1_mac_base);
#endif

	return 0;
}
#endif

/*
 * read_prodid_id: This function will read the Crestron product id pins and return 
 * 			the ID in a global variable to be consumed by the functions that 
 *			populate the environment variable for the LCD at boot time.
*/
static void read_prodid_pins(void)
{
        u8 id=0;
        u8 bits=0;
        int x=0;
	unsigned long temp;

        mxc_request_iomux(MX53_PIN_NANDF_CLE, IOMUX_CONFIG_ALT1); //prod_id_0
        mxc_request_iomux(MX53_PIN_NANDF_ALE, IOMUX_CONFIG_ALT1);
        mxc_request_iomux(MX53_PIN_NANDF_WP_B, IOMUX_CONFIG_ALT1);
        mxc_request_iomux(MX53_PIN_NANDF_RB0, IOMUX_CONFIG_ALT1);
        mxc_request_iomux(MX53_PIN_NANDF_CS0, IOMUX_CONFIG_ALT1);
        mxc_request_iomux(MX53_PIN_NANDF_WE_B, IOMUX_CONFIG_ALT1); //prod_id_5

       // Read product ID and revision ID from mx53
        temp = readl(GPIO6_BASE_ADDR + 0x4);
        temp &= 0xFFFFE07F;
        writel(temp, GPIO6_BASE_ADDR +0x4);

        // read GPIO6 data register, mask & shift      
        id=(readl(GPIO6_BASE_ADDR) & 0x00001F80) >> 7;

        gProdId = id; //Sets global variable  

}


static void setup_fec(void)
{
	volatile unsigned int reg;

	/*FEC_MDIO*/
	mxc_request_iomux(MX53_PIN_FEC_MDIO, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_MDIO, 0x1FC);
	mxc_iomux_set_input(MUX_IN_FEC_FEC_MDI_SELECT_INPUT, 0x1);

	/*FEC_MDC*/
	mxc_request_iomux(MX53_PIN_FEC_MDC, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_MDC, 0x004);

	/* FEC RXD1 */
	mxc_request_iomux(MX53_PIN_FEC_RXD1, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_RXD1, 0x180);

	/* FEC RXD0 */
	mxc_request_iomux(MX53_PIN_FEC_RXD0, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_RXD0, 0x180);

	 /* FEC TXD1 */
	mxc_request_iomux(MX53_PIN_FEC_TXD1, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_TXD1, 0x004);

	/* FEC TXD0 */
	mxc_request_iomux(MX53_PIN_FEC_TXD0, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_TXD0, 0x004);

	/* FEC TX_EN */
	mxc_request_iomux(MX53_PIN_FEC_TX_EN, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_TX_EN, 0x004);

	/* FEC TX_CLK */
	mxc_request_iomux(MX53_PIN_FEC_REF_CLK, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_REF_CLK, 0x180);

	/* FEC RX_ER */
	mxc_request_iomux(MX53_PIN_FEC_RX_ER, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_RX_ER, 0x180);

	/* FEC CRS */
	mxc_request_iomux(MX53_PIN_FEC_CRS_DV, IOMUX_CONFIG_ALT0);
	mxc_iomux_set_pad(MX53_PIN_FEC_CRS_DV, 0x180);

	/* FEC PWR_ON RJK::MX53_PIN_ATA_DATA2 from mx53_pins.h*/
		mxc_request_iomux(MX53_PIN_ATA_DATA2, IOMUX_CONFIG_ALT1);
		//mxc_iomux_set_pad(MX53_PIN_ATA_DATA2, 0x004);

		/* RJK: Set Pin as 0 */
		reg = readl(GPIO1_BASE_ADDR + 0x0);
		reg &= ~0x04;
        	writel(reg, GPIO1_BASE_ADDR + 0x0);

		/* RJK: set pin direction as output */
		reg = readl(GPIO1_BASE_ADDR + 0x4);
        	reg |= 0x04;
        	writel(reg, GPIO1_BASE_ADDR + 0x4);

		/* RJK: delay 0.1 seconds*/
		udelay(100000);

		/* RJK: Set Pin as 1 */
		reg = readl(GPIO1_BASE_ADDR + 0x0);
		reg |= 0x04;
        	writel(reg, GPIO1_BASE_ADDR + 0x0);

		/* RJK: delay 0.1 seconds*/
		udelay(100000);


	/* phy reset: gpio7-6 */
	mxc_request_iomux(MX53_PIN_ATA_DA_0, IOMUX_CONFIG_ALT1);

	reg = readl(GPIO7_BASE_ADDR + 0x0);
	reg &= ~0x40;
	writel(reg, GPIO7_BASE_ADDR + 0x0);

	reg = readl(GPIO7_BASE_ADDR + 0x4);
	reg |= 0x40;
	writel(reg, GPIO7_BASE_ADDR + 0x4);

	udelay(1000);

	reg = readl(GPIO7_BASE_ADDR + 0x0);
	reg |= 0x40;
	writel(reg, GPIO7_BASE_ADDR + 0x0);

}
#endif


#ifdef CONFIG_NET_MULTI
int board_eth_init(bd_t *bis)
{
	int rc = -ENODEV;

	return rc;
}
#endif

#ifdef CONFIG_CMD_MMC

struct fsl_esdhc_cfg esdhc_cfg[2] = {
	{MMC_SDHC1_BASE_ADDR, 1, 1},
	{MMC_SDHC3_BASE_ADDR, 1, 1},
};

#ifdef CONFIG_DYNAMIC_MMC_DEVNO
int get_mmc_env_devno(void)
{
	uint soc_sbmr = readl(SRC_BASE_ADDR + 0x4);
	return (soc_sbmr & 0x00300000) ? 1 : 0;
}
#endif

#ifdef CONFIG_EMMC_DDR_PORT_DETECT
int detect_mmc_emmc_ddr_port(struct fsl_esdhc_cfg *cfg)
{
	return (MMC_SDHC3_BASE_ADDR == cfg->esdhc_base) ? 1 : 0;
}
#endif

int esdhc_gpio_init(bd_t *bis)
{
	s32 status = 0;
	u32 index = 0;

	for (index = 0; index < CONFIG_SYS_FSL_ESDHC_NUM;
		++index) {
		switch (index) {
		case 0:
			mxc_request_iomux(MX53_PIN_SD1_CMD, IOMUX_CONFIG_ALT0);
			mxc_request_iomux(MX53_PIN_SD1_CLK, IOMUX_CONFIG_ALT0);
			mxc_request_iomux(MX53_PIN_SD1_DATA0,
						IOMUX_CONFIG_ALT0);
			mxc_request_iomux(MX53_PIN_SD1_DATA1,
						IOMUX_CONFIG_ALT0);
			mxc_request_iomux(MX53_PIN_SD1_DATA2,
						IOMUX_CONFIG_ALT0);
			mxc_request_iomux(MX53_PIN_SD1_DATA3,
						IOMUX_CONFIG_ALT0);

			mxc_iomux_set_pad(MX53_PIN_SD1_CMD, 0x1E4);
			mxc_iomux_set_pad(MX53_PIN_SD1_CLK, 0xD4);
			mxc_iomux_set_pad(MX53_PIN_SD1_DATA0, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_SD1_DATA1, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_SD1_DATA2, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_SD1_DATA3, 0x1D4);
			break;
		case 1:
			mxc_request_iomux(MX53_PIN_ATA_RESET_B,
						IOMUX_CONFIG_ALT2);
			mxc_request_iomux(MX53_PIN_ATA_IORDY,
						IOMUX_CONFIG_ALT2);
			mxc_request_iomux(MX53_PIN_ATA_DATA8,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA9,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA10,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA11,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA0,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA1,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA2,
						IOMUX_CONFIG_ALT4);
			mxc_request_iomux(MX53_PIN_ATA_DATA3,
						IOMUX_CONFIG_ALT4);

			mxc_iomux_set_pad(MX53_PIN_ATA_RESET_B, 0x1E4);
			mxc_iomux_set_pad(MX53_PIN_ATA_IORDY, 0xD4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA8, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA9, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA10, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA11, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA0, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA1, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA2, 0x1D4);
			mxc_iomux_set_pad(MX53_PIN_ATA_DATA3, 0x1D4);

			break;
		default:
			printf("Warning: you configured more ESDHC controller"
				"(%d) as supported by the board(2)\n",
				CONFIG_SYS_FSL_ESDHC_NUM);
			return status;
		}
		status |= fsl_esdhc_initialize(bis, &esdhc_cfg[index]);
	}

	return status;
}

int board_mmc_init(bd_t *bis)
{

	if (!esdhc_gpio_init(bis))
		return 0;
	else
		return -1;
}

#endif


int board_init(void)
{
#ifdef CONFIG_MFG
/* MFG firmware need reset usb to avoid host crash firstly */
#define USBCMD 0x140
	int val = readl(OTG_BASE_ADDR + USBCMD);
	val &= ~0x1; /*RS bit*/
	writel(val, OTG_BASE_ADDR + USBCMD);
#endif
	setup_boot_device();
	setup_soc_rev();

	gd->bd->bi_arch_number = MACH_TYPE_MX53_LOCO;	/* board id for linux */

	/* address of boot parameters */
	gd->bd->bi_boot_params = PHYS_SDRAM_1 + 0x100;

	read_prodid_pins();
	setup_uart();

#ifdef CONFIG_MXC_FEC
	setup_fec();
#endif

#ifdef CONFIG_I2C_MXC
	setup_i2c(CONFIG_SYS_I2C_PORT);
	i2c_init(CONFIG_SYS_I2C_SPEED, CONFIG_SYS_I2C_SLAVE);
    /* delay pmic i2c access to board_late_init()
       due to i2c_probe fail here on loco/ripley board. */
	/* Increase VDDGP voltage */
	/* setup_pmic_voltages(); */
	/* Switch to 1GHZ */
	/* clk_config(CONFIG_REF_CLK_FREQ, 1000, CPU_CLK); */
#endif

#if defined(CONFIG_DWC_AHSATA)
	setup_sata_device();
#endif

	return 0;
}


#ifdef CONFIG_ANDROID_RECOVERY
struct reco_envs supported_reco_envs[BOOT_DEV_NUM] = {
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
	{
	 .cmd = CONFIG_ANDROID_RECOVERY_BOOTCMD_MMC,
	 .args = CONFIG_ANDROID_RECOVERY_BOOTARGS_MMC,
	 },
	{
	 .cmd = CONFIG_ANDROID_RECOVERY_BOOTCMD_MMC,
	 .args = CONFIG_ANDROID_RECOVERY_BOOTARGS_MMC,
	 },
	{
	 .cmd = NULL,
	 .args = NULL,
	 },
};

int check_recovery_cmd_file(void)
{
	disk_partition_t info;
	ulong part_length;
	int filelen;
	char *env;

	/* For test only */
	/* When detecting android_recovery_switch,
	 * enter recovery mode directly */
	env = getenv("android_recovery_switch");
	if (!strcmp(env, "1")) {
		printf("Env recovery detected!\nEnter recovery mode!\n");
		return 1;
	}

	printf("Checking for recovery command file...\n");
	switch (get_boot_device()) {
	case MMC_BOOT:
	case SD_BOOT:
		{
			block_dev_desc_t *dev_desc = NULL;
			struct mmc *mmc = find_mmc_device(0);

			dev_desc = get_dev("mmc", 0);

			if (NULL == dev_desc) {
				puts("** Block device MMC 0 not supported\n");
				return 0;
			}

			mmc_init(mmc);

			if (get_partition_info(dev_desc,
					CONFIG_ANDROID_CACHE_PARTITION_MMC,
					&info)) {
				printf("** Bad partition %d **\n",
					CONFIG_ANDROID_CACHE_PARTITION_MMC);
				return 0;
			}

			part_length = ext2fs_set_blk_dev(dev_desc,
					CONFIG_ANDROID_CACHE_PARTITION_MMC);
			if (part_length == 0) {
				printf("** Bad partition - mmc 0:%d **\n",
					CONFIG_ANDROID_CACHE_PARTITION_MMC);
				ext2fs_close();
				return 0;
			}

			if (!ext2fs_mount(part_length)) {
				printf("** Bad ext2 partition or "
					"disk - mmc 0:%d **\n",
					CONFIG_ANDROID_CACHE_PARTITION_MMC);
				ext2fs_close();
				return 0;
			}

			filelen = ext2fs_open(CONFIG_ANDROID_RECOVERY_CMD_FILE);

			ext2fs_close();
		}
		break;
	case NAND_BOOT:
		return 0;
		break;
	case SPI_NOR_BOOT:
		return 0;
		break;
	case UNKNOWN_BOOT:
	default:
		return 0;
		break;
	}

	return (filelen > 0) ? 1 : 0;

}
#endif

#ifdef CONFIG_I2C_MXC
/* Note: udelay() is not accurate for i2c timing */
static void __udelay(int time)
{
	int i, j;

	for (i = 0; i < time; i++) {
		for (j = 0; j < 200; j++) {
			asm("nop");
			asm("nop");
		}
	}
}
#define I2C1_SDA_GPIO5_26_BIT_MASK  (1 << 26)
#define I2C1_SCL_GPIO5_27_BIT_MASK  (1 << 27)
#define I2C2_SCL_GPIO4_12_BIT_MASK  (1 << 12)
#define I2C2_SDA_GPIO4_13_BIT_MASK  (1 << 13)
#define I2C3_SCL_GPIO1_3_BIT_MASK   (1 << 3)
#define I2C3_SDA_GPIO1_6_BIT_MASK   (1 << 6)
static void mx53_i2c_gpio_scl_direction(int bus, int output)
{
	u32 reg;

	switch (bus) {
	case 1:
		mxc_request_iomux(MX53_PIN_CSI0_D9, IOMUX_CONFIG_ALT1);
		reg = readl(GPIO5_BASE_ADDR + GPIO_GDIR);
		if (output)
			reg |= I2C1_SCL_GPIO5_27_BIT_MASK;
		else
			reg &= ~I2C1_SCL_GPIO5_27_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + GPIO_GDIR);
		break;
	case 2:
		mxc_request_iomux(MX53_PIN_KEY_COL3, IOMUX_CONFIG_ALT1);
		reg = readl(GPIO4_BASE_ADDR + GPIO_GDIR);
		if (output)
			reg |= I2C2_SCL_GPIO4_12_BIT_MASK;
		else
			reg &= ~I2C2_SCL_GPIO4_12_BIT_MASK;
		writel(reg, GPIO4_BASE_ADDR + GPIO_GDIR);
		break;
	case 3:
		mxc_request_iomux(MX53_PIN_GPIO_3, IOMUX_CONFIG_ALT1);
		reg = readl(GPIO1_BASE_ADDR + GPIO_GDIR);
		if (output)
			reg |= I2C3_SCL_GPIO1_3_BIT_MASK;
		else
			reg &= I2C3_SCL_GPIO1_3_BIT_MASK;
		writel(reg, GPIO1_BASE_ADDR + GPIO_GDIR);
		break;
	}
}

/* set 1 to output, sent 0 to input */
static void mx53_i2c_gpio_sda_direction(int bus, int output)
{
	u32 reg;

	switch (bus) {
	case 1:
		mxc_request_iomux(MX53_PIN_CSI0_D8, IOMUX_CONFIG_ALT1);

		reg = readl(GPIO5_BASE_ADDR + GPIO_GDIR);
		if (output) {
			mxc_iomux_set_pad(MX53_PIN_CSI0_D8,
					  PAD_CTL_ODE_OPENDRAIN_ENABLE |
					  PAD_CTL_DRV_HIGH | PAD_CTL_100K_PU);
			reg |= I2C1_SDA_GPIO5_26_BIT_MASK;
		} else
			reg &= ~I2C1_SDA_GPIO5_26_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + GPIO_GDIR);
		break;
	case 2:
		mxc_request_iomux(MX53_PIN_KEY_ROW3, IOMUX_CONFIG_ALT1);

		mxc_iomux_set_pad(MX53_PIN_KEY_ROW3,
				  PAD_CTL_SRE_FAST |
				  PAD_CTL_ODE_OPENDRAIN_ENABLE |
				  PAD_CTL_DRV_HIGH | PAD_CTL_100K_PU |
				  PAD_CTL_HYS_ENABLE);

		reg = readl(GPIO4_BASE_ADDR + GPIO_GDIR);
		if (output)
			reg |= I2C2_SDA_GPIO4_13_BIT_MASK;
		else
			reg &= ~I2C2_SDA_GPIO4_13_BIT_MASK;
		writel(reg, GPIO4_BASE_ADDR + GPIO_GDIR);
	case 3:
		mxc_request_iomux(MX53_PIN_GPIO_6, IOMUX_CONFIG_ALT1);
		mxc_iomux_set_pad(MX53_PIN_GPIO_6,
				  PAD_CTL_PUE_PULL | PAD_CTL_PKE_ENABLE |
				  PAD_CTL_DRV_HIGH | PAD_CTL_360K_PD |
				  PAD_CTL_HYS_ENABLE);
		reg = readl(GPIO1_BASE_ADDR + GPIO_GDIR);
		if (output)
			reg |= I2C3_SDA_GPIO1_6_BIT_MASK;
		else
			reg &= ~I2C3_SDA_GPIO1_6_BIT_MASK;
		writel(reg, GPIO1_BASE_ADDR + GPIO_GDIR);
	default:
		break;
	}
}

/* set 1 to high 0 to low */
static void mx53_i2c_gpio_scl_set_level(int bus, int high)
{
	u32 reg;
	switch (bus) {
	case 1:
		reg = readl(GPIO5_BASE_ADDR + GPIO_DR);
		if (high)
			reg |= I2C1_SCL_GPIO5_27_BIT_MASK;
		else
			reg &= ~I2C1_SCL_GPIO5_27_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + GPIO_DR);
		break;
	case 2:
		reg = readl(GPIO4_BASE_ADDR + GPIO_DR);
		if (high)
			reg |= I2C2_SCL_GPIO4_12_BIT_MASK;
		else
			reg &= ~I2C2_SCL_GPIO4_12_BIT_MASK;
		writel(reg, GPIO4_BASE_ADDR + GPIO_DR);
		break;
	case 3:
		reg = readl(GPIO1_BASE_ADDR + GPIO_DR);
		if (high)
			reg |= I2C3_SCL_GPIO1_3_BIT_MASK;
		else
			reg &= ~I2C3_SCL_GPIO1_3_BIT_MASK;
		writel(reg, GPIO1_BASE_ADDR + GPIO_DR);
		break;
	}
}

/* set 1 to high 0 to low */
static void mx53_i2c_gpio_sda_set_level(int bus, int high)
{
	u32 reg;

	switch (bus) {
	case 1:
		reg = readl(GPIO5_BASE_ADDR + GPIO_DR);
		if (high)
			reg |= I2C1_SDA_GPIO5_26_BIT_MASK;
		else
			reg &= ~I2C1_SDA_GPIO5_26_BIT_MASK;
		writel(reg, GPIO5_BASE_ADDR + GPIO_DR);
		break;
	case 2:
		reg = readl(GPIO4_BASE_ADDR + GPIO_DR);
		if (high)
			reg |= I2C2_SDA_GPIO4_13_BIT_MASK;
		else
			reg &= ~I2C2_SDA_GPIO4_13_BIT_MASK;
		writel(reg, GPIO4_BASE_ADDR + GPIO_DR);
		break;
	case 3:
		reg = readl(GPIO1_BASE_ADDR + GPIO_DR);
		if (high)
			reg |= I2C3_SDA_GPIO1_6_BIT_MASK;
		else
			reg &= ~I2C3_SDA_GPIO1_6_BIT_MASK;
		writel(reg, GPIO1_BASE_ADDR + GPIO_DR);
		break;
	}
}

static int mx53_i2c_gpio_check_sda(int bus)
{
	u32 reg;
	int result = 0;

	switch (bus) {
	case 1:
		reg = readl(GPIO5_BASE_ADDR + GPIO_PSR);
		result = !!(reg & I2C1_SDA_GPIO5_26_BIT_MASK);
		break;
	case 2:
		reg = readl(GPIO4_BASE_ADDR + GPIO_PSR);
		result = !!(reg & I2C2_SDA_GPIO4_13_BIT_MASK);
		break;
	case 3:
		reg = readl(GPIO1_BASE_ADDR + GPIO_PSR);
		result = !!(reg & I2C3_SDA_GPIO1_6_BIT_MASK);
		break;
	}

	return result;
}


 /* Random reboot cause i2c SDA low issue:
  * the i2c bus busy because some device pull down the I2C SDA
  * line. This happens when Host is reading some byte from slave, and
  * then host is reset/reboot. Since in this case, device is
  * controlling i2c SDA line, the only thing host can do this give the
  * clock on SCL and sending NAK, and STOP to finish this
  * transaction.
  *
  * How to fix this issue:
  * detect if the SDA was low on bus send 8 dummy clock, and 1
  * clock + NAK, and STOP to finish i2c transaction the pending
  * transfer.
  */
int i2c_bus_recovery(void)
{
	int i, bus, result = 0;

	for (bus = 1; bus <= 3; bus++) {
		mx53_i2c_gpio_sda_direction(bus, 0);

		if (mx53_i2c_gpio_check_sda(bus) == 0) {
			printf("i2c: I2C%d SDA is low, start i2c recovery...\n", bus);
			mx53_i2c_gpio_scl_direction(bus, 1);
			mx53_i2c_gpio_scl_set_level(bus, 1);
			__udelay(10000);

			for (i = 0; i < 9; i++) {
				mx53_i2c_gpio_scl_set_level(bus, 1);
				__udelay(5);
				mx53_i2c_gpio_scl_set_level(bus, 0);
				__udelay(5);
			}

			/* 9th clock here, the slave should already
			   release the SDA, we can set SDA as high to
			   a NAK.*/
			mx53_i2c_gpio_sda_direction(bus, 1);
			mx53_i2c_gpio_sda_set_level(bus, 1);
			__udelay(1); /* Pull up SDA first */
			mx53_i2c_gpio_scl_set_level(bus, 1);
			__udelay(5); /* plus pervious 1 us */
			mx53_i2c_gpio_scl_set_level(bus, 0);
			__udelay(5);
			mx53_i2c_gpio_sda_set_level(bus, 0);
			__udelay(5);
			mx53_i2c_gpio_scl_set_level(bus, 1);
			__udelay(5);
			/* Here: SCL is high, and SDA from low to high, it's a
			 * stop condition */
			mx53_i2c_gpio_sda_set_level(bus, 1);
			__udelay(5);

			mx53_i2c_gpio_sda_direction(bus, 0);
			if (mx53_i2c_gpio_check_sda(bus) == 1)
				printf("I2C%d Recovery success\n", bus);
			else {
				printf("I2C%d Recovery failed, I2C1 SDA still low!!!\n", bus);
				result |= 1 << bus;
			}
		}

		/* configure back to i2c */
		switch (bus) {
		case 1:
			setup_i2c(I2C1_BASE_ADDR);
			break;
		case 2:
			setup_i2c(I2C2_BASE_ADDR);
			break;
		case 3:
			setup_i2c(I2C3_BASE_ADDR);
			break;
		}
	}

	return result;
}
#endif

/* restore VUSB 2V5 active after suspend */
#define BUCKPERI_RESTORE_SW_STEP   (0x55)
/* restore VUSB 2V5 power supply after suspend */
#define SUPPLY_RESTORE_VPERISW_EN  (0x20)

#define DA9052_STATUSA_REG			1
#define DA9052_STATUSB_REG			2
#define DA9052_STATUSC_REG			3
#define DA9052_STATUSD_REG			4
#define DA9052_EVENTA_REG			5
#define DA9052_EVENTB_REG			6
#define DA9052_EVENTC_REG			7
#define DA9052_EVENTD_REG			8
#define DA9052_FAULTLOG_REG			9
#define DA9052_IRQMASKA_REG			10
#define DA9052_IRQMASKB_REG			11
#define DA9052_IRQMASKC_REG			12
#define DA9052_IRQMASKD_REG			13
#define DA9052_CONTROLA_REG			14
#define DA9052_CONTROLB_REG			15
#define DA9052_CONTROLC_REG			16
#define DA9052_CONTROLD_REG			17
#define DA9052_ID01_REG				29
#define DA9052_ID23_REG				30
#define DA9052_ID45_REG				31
#define DA9052_ID67_REG				32
#define DA9052_ID89_REG				33
#define DA9052_ID1011_REG			34
#define DA9052_ID1213_REG			35
#define DA9052_ID1415_REG			36
#define DA9052_ID1617_REG			37
#define DA9052_ID1819_REG			38
#define DA9052_ID2021_REG			39
#define DA9052_SEQSTATUS_REG		40
#define DA9052_SEQA_REG				41
#define DA9052_SEQB_REG				42
#define DA9052_SEQTIMER_REG			43
#define DA9052_SUPPLY_REG           60


#define DA9052_FAULTLOG_VDDFAULT		(1<<1)
#define DA9052_CONTROLC_PMFB2PIN		(1<<1)
#define DA9052_IRQMASKA_MDCINVLD		(1<<0)
#define DA9052_CONTROLA_SYSEN			(1<<0)
#define DA9052_STATUSA_DCINDET			(1<<3)

/* FAULT LOG REGISTER */
#define DA9052_FAULTLOG_WAITSET			(1<<7)
#define DA9052_FAULTLOG_NSDSET			(1<<6)
#define DA9052_FAULTLOG_KEYSHUT			(1<<5)
#define DA9052_FAULTLOG_TEMPOVER		(1<<3)
#define DA9052_FAULTLOG_VDDSTART		(1<<2)
#define DA9052_FAULTLOG_VDDFAULT		(1<<1)
#define DA9052_FAULTLOG_TWDERROR		(1<<0)


int board_late_init(void)
{
	uchar value;
	unsigned char buf[4] = { 0 };
	int retries = 10, ret = -1;
	int battery_level, battery_voltage;
	char *env;
	char env_str[50];
	
#ifdef CONFIG_I2C_MXC
	/* first recovery I2C bus in case other device in some i2c
	 * transcation */
	i2c_bus_recovery();
#endif

	if (!i2c_probe(0x8)) {
		if (i2c_read(0x8, 24, 1, &buf[0], 3)) {
			printf("%s:i2c_read:error\n", __func__);
			return -1;
		}
		/* increase VDDGP as 1.25V for 1GHZ on SW1 */
		buf[2] = 0x30;
		if (i2c_write(0x8, 24, 1, buf, 3)) {
			printf("%s:i2c_write:error\n", __func__);
			return -1;
		}
		if (i2c_read(0x8, 25, 1, &buf[0], 3)) {
			printf("%s:i2c_read:error\n", __func__);
			return -1;
		}
		/* increase VCC as 1.3V on SW2 */
		buf[2] = 0x34;
		if (i2c_write(0x8, 25, 1, buf, 3)) {
			printf("%s:i2c_write:error\n", __func__);
			return -1;
		}
		/*change global reset time as 4s*/
		if (i2c_read(0x8, 15, 1, &buf[0], 3)) {
			printf("%s:i2c_read:error\n", __func__);
			return -1;
		}
		buf[1] |= 0x1;
		buf[1] &= ~0x2;
		if (i2c_write(0x8, 15, 1, buf, 3)) {
			printf("%s:i2c_write:error\n", __func__);
			return -1;
		}

		/* set up rev #1 for loco/ripley board */
		setup_board_rev(get_board_rev_from_fuse());
		/* Switch to 1GHZ */
		clk_config(CONFIG_REF_CLK_FREQ, 1000, CPU_CLK);
	} else if (!i2c_probe(0x48)) {
		/* increase VDDGP as 1.25V for 1GHZ */
		value = 0x5e;
		do {
			if (0 != i2c_write_check(0x48, 0x2e, 1, &value, 1)) {
				printf("da9052_i2c_is_connected - i2c write failed.....\n");
			} else {
				printf("da9052_i2c_is_connected - i2c write success....\n");
				ret = 0;
			}
		} while (ret != 0 && retries--);
		i2c_read(0x48, 60, 1, &value, 1);
		value |= 0x1;
		i2c_write(0x48, 60, 1, &value, 1);
		/* restore VUSB_2V5 when reset from suspend state */
		value = BUCKPERI_RESTORE_SW_STEP;
		i2c_write(0x48, DA9052_ID1213_REG, 1, &value, 1);
		i2c_read(0x48, DA9052_SUPPLY_REG, 1, &value, 1);
		value |= SUPPLY_RESTORE_VPERISW_EN;
		i2c_write(0x48, DA9052_SUPPLY_REG, 1, &value, 1);

		/* set up rev #0 for loco/da9053 board */
		setup_board_rev(0);
		/* Switch to 1GHZ */
		clk_config(CONFIG_REF_CLK_FREQ, 1000, CPU_CLK);
	} else {
		printf("Error: Dont't found mc34708 or da9052 on board.\n");
	}

#ifdef CONFIG_HW_WATCHDOG
	/* Implicitly used from arm-cortexa8/mx53/watchdog.c */
	hw_watchdog_init();
#endif
	/* checkboard() is called before board_late_init() in the init_sequence.
	 * g_boot_reason is already set.*/
	setenv("boot_status", g_boot_reason);

	// Store the previous boot/suspend information so that it can be printed.
	// We need to store the previous information since, if that boot failed, the one that
	// prints will have succeded and it's the failure that we are interested in.
	env = getenv("RebootResult");
	sprintf(env_str, "%s", env);
	setenv("LastRebootResult", env_str);

	env = getenv("BootBatterySOC");
	sprintf(env_str, "%s", env);
	setenv("LastBootBatterySOC", env_str);

	env = getenv("BootBatteryVoltage");
	sprintf(env_str, "%s", env);
	setenv("LastBootBatteryVoltage", env_str);

	env = getenv("CriticalBatteryVoltage");
	sprintf(env_str, "%s", env);
	setenv("LastCriticalBatteryVoltage", env_str);

	env = getenv("CriticalBatterySOC");
	sprintf(env_str, "%s", env);
	setenv("LastCriticalBatterySOC", env_str);

	env = getenv("PMICFaultRegister");
	sprintf(env_str, "%s", env);
	setenv("LastPMICFaultRegister", env_str);


	// Check the battery level to see if there is enough capacity to run or if it needs to
	// go directly into shutdown mode.

	battery_level = CRITICAL_BATTERY_LEVEL;
	battery_voltage = BOOT_VOLTAGE_THRESHOLD;

	if (strcmp(g_boot_reason, "RST") == 0)
		battery_voltage = LOADED_BOOT_VOLTAGE_THRESHOLD;	// we got here from warm reset, use loaded battery voltage
	if (strcmp(g_boot_reason, "WDOG") == 0)
		battery_voltage = LOADED_BOOT_VOLTAGE_THRESHOLD;	// we got here from warm reset, use loaded battery voltage

	if (!BatteryGood(battery_level, battery_voltage))
	{
		printf("Battery Level too low.  Going into shut down.  Please dock panel.\r\n");
		OALDa9052ShutDown();
	}

	saveenv();			// preserve boot values in environment

	return 0;
}

int checkboard(void)
{
	short val;
	printf("Board: ");
	printf("MX53-CRESTRON ");
	switch (get_board_rev_from_fuse()) {
	case 0x3:
		printf("Rev. B\n");
		break;
	case 0x1:
	default:
		printf("Rev. A\n");
		break;
	}

	/* Write the reboot reason to an environment variable. */
	val = __REG(SRC_BASE_ADDR + 0x8);
	switch (val) {
	case 0x0001:
		g_boot_reason = "POR";
		break;
	case 0x0009:
		g_boot_reason = "RST";
		break;
	case 0x0010:
	case 0x0011:
		g_boot_reason = "WDOG";
		break;
	default:
		printf("unknown");
	}
	printf("Boot Reason: [0x%04x]=[%s]\n", val, g_boot_reason);

	printf("Boot Device: ");
	switch (get_boot_device()) {
	case WEIM_NOR_BOOT:
		printf("NOR\n");
		break;
	case ONE_NAND_BOOT:
		printf("ONE NAND\n");
		break;
	case PATA_BOOT:
		printf("PATA\n");
		break;
	case SATA_BOOT:
		printf("SATA\n");
		break;
	case I2C_BOOT:
		printf("I2C\n");
		break;
	case SPI_NOR_BOOT:
		printf("SPI NOR\n");
		break;
	case SD_BOOT:
		printf("SD\n");
		break;
	case MMC_BOOT:
		printf("MMC\n");
		break;
	case NAND_BOOT:
		printf("NAND\n");
		break;
	case UNKNOWN_BOOT:
	default:
		printf("UNKNOWN\n");
		break;
	}
	return 0;
}



//-----------------------------------------------------------------------------
//
// Function: BatteryGood
//
//  This function returns TRUE if the battery should be good enough to allow the
//	unit to boot.
//
// Parameters:
//		dwBootMargin - percentage above "battery critical" level that battery needs to be
//			if we are doing a warm reboot.
//		dwBootVoltageThreshold - minimum battery voltage (in mV) that the battery needs to
//			be if we are doing a cold reboot.
//
// Returns:
//      BOOL - True if unit can be allowed to boot
//
//	Author: John Strowe
//	Date:	Mar. 1 2013
//-----------------------------------------------------------------------------


static int BatteryGood (int dwBootSOCThreshold, int dwBootVoltageThreshold)
{
	int bBatLevelOK = 1;
	unsigned char FaultLog;
	unsigned char onAC;
	unsigned char rsoc = 0;
	unsigned char voltH, voltL;
	int volt = 0;
	int maxloops;
	int BatteryLevelLimit;
	unsigned char value;
	char *env;
	char env_string[100];
	int gas_gauge_ok = 0;


	setenv("RebootResult", "reboot_OK");

	//only 300, 600 and 900 have battery currently
	if ( (gProdId != TSR300) && (gProdId != TST600) && (gProdId != TST900) )
		return 1;

	if(gProdId != TST900){
		if(!i2c_probe(0x36))
			gas_gauge_ok = 1;
		else
			printf("300/600 gas gauge not responding.\r\n");
	}else{
		if(!i2c_probe(0x55))
			gas_gauge_ok = 1;
		else
			printf("900 gas gauge not responding.\r\n");
	}

	if (!i2c_probe(0x48) && gas_gauge_ok) {
		// Print the PMIC fault log so that we know what caused the reset to happen.
		// The watchdog driver will also print this information and clear the faults.
		// However, the watchdog will not run if the battery is low.
		i2c_read(0x48, DA9052_FAULTLOG_REG, 1, &FaultLog, 1);
		printf("PMIC faultlog register = 0x%x.\r\n",FaultLog);
		if(FaultLog & DA9052_FAULTLOG_TWDERROR)
			printf("PMIC watchdog time violation.\r\n");
		if(FaultLog & DA9052_FAULTLOG_VDDFAULT)
			printf("PMIC VDDOUT undervoltage detected.\r\n");
		if(FaultLog & DA9052_FAULTLOG_VDDSTART)
			printf("PMIC VDDOUT undervoltage detected within 10 seconds of reset.\r\n");
		if(FaultLog & DA9052_FAULTLOG_TEMPOVER)
			printf("PMIC junction over temperature.\r\n");
		if(FaultLog & DA9052_FAULTLOG_KEYSHUT)
			printf("PMIC nONKEY held asserted (power button).\r\n");
		if(FaultLog & DA9052_FAULTLOG_NSDSET)
			printf("PMIC nSHUTDOWN asserted.\r\n");
		if(FaultLog & DA9052_FAULTLOG_WAITSET)
			printf("PMIC ID WAIT_STEP timeout.\r\n");
		if(!(FaultLog))
			printf("PMIC - no fault bits set.\r\n");
		value = 0xFF;
		i2c_write(0x48, DA9052_FAULTLOG_REG, 1, &value, 1);		// clear the fault log for next time

		//check if on AC
		i2c_read(0x48, DA9052_STATUSA_REG, 1, &value, 1);
		printf("DA9052_STATUSA_REG = 0x%02X\r\n",value);
		//DCIN_DET is bit 3
		//if on AC , skip battery check, otherwise check battery
		if(value & DA9052_STATUSA_DCINDET)
		{
			printf("On AC\r\n");
			if ((gProdId == TSR300) || (gProdId == TST600))
			{
				//i2c gas gauge address = 0x36
				//DS2786B_RELCAP = 0x02
				i2c_read(0x36, 0x02, 1, &rsoc, 1);
				rsoc = rsoc / 2;
				//0x0C = voltage high bits
				//0x0D = voltage low bits
				i2c_read(0x36, 0x0C, 1, &voltH, 1);
				i2c_read(0x36, 0x0D, 1, &voltL, 1);
				//right shift 3 since lowest 3 bits unused
				volt = (((voltH << 8) | voltL) >> 3);
				//having trouble with * 1.22, do 1*volt + .25*volt -.03125*volt = 1.21875; //1.22 mv per step (from data sheet)
				volt = volt + (volt >> 2) - (volt >> 5);
			}
			else	// TST900
			{
				//i2c gas gauge address = 0x55
				//BQ27200_RSOC = 0x0B
				i2c_read(0x55, 0x0B, 1, &rsoc, 1);
				//0x09 = voltage high bits
				//0x08 = voltage low bits
				i2c_read(0x55, 0x09, 1, &voltH, 1);
				i2c_read(0x55, 0x08, 1, &voltL, 1);
				volt = ((voltH << 8) | voltL);
			}
			printf("Battery Level: %i Voltage: %i\r\n",rsoc, volt);

			bBatLevelOK = 1;					// on AC, do not worry about battery level
		}
		else
		{
			printf("On Battery\n");
			if ((gProdId == TSR300) || (gProdId == TST600))
			{
				//i2c gas gauge address = 0x36
				//DS2786B_RELCAP = 0x02
				i2c_read(0x36, 0x02, 1, &rsoc, 1);					// get SOC from gas gauge
				rsoc = rsoc / 2;
			}
			else	// TST900
			{
				//i2c gas gauge address = 0x55
				//BQ27200_RSOC = 0x0B
				i2c_read(0x55, 0x0B, 1, &rsoc, 1);
			}

			// We are booting on battery power. We will estimate whether or not we have enough
			// power to allow the unit to boot to the point where the application can monitor the
			// battery capacity.
			//
			// If doing a warm reboot on battery, determine if the battery is above the application's critical battery level
			// plus a buffer passed in.
			// UNLESS:
			// The PMIC fault log has VDD_FAULT_BIT set which indicates that either the battery switch was cycled
			// or that the battery collapsed unexpectedly which means that the SOC is not accurate.

			if ((FaultLog & DA9052_FAULTLOG_VDDFAULT)) // battery collapsed or power-cycle reboot - not warm reboot
			{
				// cold boot or battery collapsed, look at battery voltage only

				for(maxloops = 0; maxloops < 3; maxloops++)
				{
					if ((gProdId == TSR300) || (gProdId == TST600))
					{
						//0x0C = voltage high bits
						//0x0D = voltage low bits
						i2c_read(0x36, 0x0C, 1, &voltH, 1);
						i2c_read(0x36, 0x0D, 1, &voltL, 1);
						//right shift 3 since lowest 3 bits unused
						volt = (((voltH << 8) | voltL) >> 3);
						//having trouble with * 1.22, do 1*volt + .25*volt -.03125*volt = 1.21875; //1.22 mv per step (from data sheet)
						volt = volt + (volt >> 2) - (volt >> 5);
					}
					else
					{
						//0x09 = voltage high bits
						//0x08 = voltage low bits
						i2c_read(0x55, 0x09, 1, &voltH, 1);
						i2c_read(0x55, 0x08, 1, &voltL, 1);
						volt = ((voltH << 8) | voltL);
					}
					if(volt < dwBootVoltageThreshold)
					{
						//retry. could be invalid read.
					}
					else //good voltage
						break;
					udelay(1000000);	 //wait 1 s
				}
				printf("Checking battery voltage: Level: %i, Voltage: %i\r\n",rsoc,volt);
				if(volt < dwBootVoltageThreshold)
				{
					bBatLevelOK = 0;
					setenv("RebootResult", "reboot_failed_voltage_low");
				}
				printf("Critical voltage: %i, bBatLevelOK = 0x%X\r\n",dwBootVoltageThreshold, bBatLevelOK);
			}
			else
			{
				// warm boot and reboot was not due to the battery collapsing, look at SOC

				printf("Checking battery SOC: Level: %i\r\n",rsoc);
				if(rsoc < dwBootSOCThreshold)
				{
					bBatLevelOK = 0;
					setenv("RebootResult", "reboot_failed_SOC_low");
				}
				printf("Critical Level: %i, bBatLevelOK = 0x%X\r\n",dwBootSOCThreshold, bBatLevelOK);
			}
		}
	}

	sprintf(env_string, "0x%02X", FaultLog);
	setenv("PMICFaultRegister", env_string);
	// Save SOC, voltage and limit values for logging later
	sprintf(env_string, "%d", rsoc);
	setenv("BootBatterySOC", env_string);
	sprintf(env_string, "%d", volt);
	setenv("BootBatteryVoltage", env_string);
	sprintf(env_string, "%d", dwBootVoltageThreshold);
	setenv("CriticalBatteryVoltage", env_string);
	sprintf(env_string, "%d", dwBootSOCThreshold);
	setenv("CriticalBatterySOC", env_string);

	return bBatLevelOK;
}



//-----------------------------------------------------------------------------
//
//  Function: OALDa9052ShutDown
//
//  Puts the DA9053 into full shutdown mode for absolute minimum battery drain.
//	The CPU is not powered during shutdown and will not recover its previous state.
//	There is no return from this routine since the CPU state and memory contents will
//	be destroyed by the removal of power.
//
//	This routine is intended to be called when the battery charge becomes critically low
//	so that the cells will not be damaged by over-discharge.
//
//	There are only two ways to recover from this state: Cycling battery power and putting
//	the unit back on the charger. The DCIN signal from the charger is the only input
//	that the PMIC will recognize as a wakeup input.
//
//	When the PIMC wakes up or the battery power is cycled, the PIMIC will assert the reset
//	signal so that the CPU will go through the full reset initialization.
//
//  Parameters:
//      NONE
//
//  Returns:
//      NONE
//
//-----------------------------------------------------------------------------
static void OALDa9052ShutDown(void)
{
	unsigned char value;

	saveenv();			// preserve boot values in environment before shutting down
	udelay(1000000);	 //wait 1 s to make sure that the save has finished

#ifdef CONFIG_I2C_MXC
	/* first recovery I2C bus in case other device in some i2c
	 * transcation */
	i2c_bus_recovery();
#endif

	printf("Unit doing full powerdown, dock to restart unit.\r\n");

	value = 0xEF;
	i2c_write(0x48, DA9052_CONTROLA_REG, 1, &value, 1);
	value = 0x3E;
	i2c_write(0x48, DA9052_CONTROLB_REG, 1, &value, 1);

	// Set PWR_UP_GP_FB2 as state of "POWER1" domain of PMIC. This lets it go low in suspend mode.
	// That shuts down the DCDC_3V3 switching supply.
	i2c_read(0x48, DA9052_CONTROLC_REG, 1, &value, 1);
	value &= (unsigned char)~DA9052_CONTROLC_PMFB2PIN;
	i2c_write(0x48, DA9052_CONTROLC_REG, 1, &value, 1);

	value = 0x10;
	i2c_write(0x48, DA9052_CONTROLD_REG, 1, &value, 1);

	// Assert NRESET output of PMIC during shutdown mode.
	//	This assures that the CPU will execute a cold reset when resuming.
	// All supplies programmed according to the PMIC's OTP.
	value = 0x23;
	i2c_write(0x48, DA9052_ID01_REG, 1, &value, 1);

	// Reprogram the rest of the sequencer to the PMIC's default for powering back up.
	value = 0xA5;
	i2c_write(0x48, DA9052_ID23_REG, 1, &value, 1);
	value = 0x56;
	i2c_write(0x48, DA9052_ID45_REG, 1, &value, 1);
	value = 0x63;
	i2c_write(0x48, DA9052_ID67_REG, 1, &value, 1);
	value = 0xA3;
	i2c_write(0x48, DA9052_ID89_REG, 1, &value, 1);
	value = 0x13;
	i2c_write(0x48, DA9052_ID1011_REG, 1, &value, 1);
	value = 0x55;
	i2c_write(0x48, DA9052_ID1213_REG, 1, &value, 1);
	value = 0x24;
	i2c_write(0x48, DA9052_ID1415_REG, 1, &value, 1);
	value = 0x35;
	i2c_write(0x48, DA9052_ID1617_REG, 1, &value, 1);
	value = 0xAA;
	i2c_write(0x48, DA9052_ID1819_REG, 1, &value, 1);
	value = 0x00;
	i2c_write(0x48, DA9052_ID2021_REG, 1, &value, 1);
	value = 0x64;
	i2c_write(0x48, DA9052_SEQA_REG, 1, &value, 1);
	value = 0x1B;
	i2c_write(0x48, DA9052_SEQB_REG, 1, &value, 1);
	value = 0x00;
	i2c_write(0x48, DA9052_SEQTIMER_REG, 1, &value, 1);


	// Set up which inputs will allow the PMIC to wake up.
	value = (unsigned char)~DA9052_IRQMASKA_MDCINVLD;
	i2c_write(0x48, DA9052_IRQMASKA_REG, 1, &value, 1);
	value = 0xFF;
	i2c_write(0x48, DA9052_IRQMASKB_REG, 1, &value, 1);
	value = 0xFF;
	i2c_write(0x48, DA9052_IRQMASKC_REG, 1, &value, 1);
	value = 0xFF;
	i2c_write(0x48, DA9052_IRQMASKD_REG, 1, &value, 1);

	// Clear the event registers
	value = 0xFF;
	i2c_write(0x48, DA9052_EVENTA_REG, 1, &value, 1);
	value = 0xFF;
	i2c_write(0x48, DA9052_EVENTB_REG, 1, &value, 1);
	value = 0xFF;
	i2c_write(0x48, DA9052_EVENTC_REG, 1, &value, 1);
	value = 0xFF;
	i2c_write(0x48, DA9052_EVENTD_REG, 1, &value, 1);

	// Set PMIC into standby mode with an I2C command. There is a 35uS delay before the PMIC begins powering down.
	i2c_read(0x48, DA9052_CONTROLA_REG, 1, &value, 1);
	value &= (unsigned char)~DA9052_CONTROLA_SYSEN;
	i2c_write(0x48, DA9052_CONTROLA_REG, 1, &value, 1);

	udelay(1000000);								// Wait here until the supplies collapse

}



